package weibo4j;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import weibo4j.http.Response;
import weibo4j.org.json.JSONException;
import weibo4j.org.json.JSONObject;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * A data class representing one single status of a user.
 * 
 * @author Yusuke Yamamoto - yusuke at mac.com
 */
public class Comment extends WeiboResponse implements java.io.Serializable
{

	private Date createdAt;
	private String id;
	private String text;
	private String source;
	private String sourceUrl;
	private boolean isTruncated;
	private long inReplyToStatusId;
	private int inReplyToUserId;
	private boolean isFavorited;
	private String inReplyToScreenName;
	private double latitude = -1;
	private double longitude = -1;

	private RetweetDetails retweetDetails;
	private Status status;
	private static final long serialVersionUID = 1608000492860584608L;

	/* package */Comment(Response res, Weibo twitter) throws WeiboException
	{
		super(res);

		Element elem = res.asDocument().getDocumentElement();
		init(res, elem, twitter);
	}

	/* package */Comment(Response res, Element elem, Weibo twitter)
			throws WeiboException
	{

		super(res);

		init(res, elem, twitter);
	}

	public Comment(String str) throws WeiboException, JSONException
	{
		// StatusStream uses this constructor
		super();
		JSONObject json = new JSONObject(str);
		id = json.getString("id");
		text = json.getString("text");
		source = json.getString("source");
		createdAt = parseDate(json.getString("created_at"),
				"EEE MMM dd HH:mm:ss z yyyy");

		user = new User(json.getJSONObject("user"));
	}

	private void init(Response res, Element elem, Weibo twitter)
			throws WeiboException
	{
		try
		{
			ensureRootNodeNameIs("comment", elem);

			user = new User(res, (Element) elem.getElementsByTagName("user")
					.item(0), twitter);
			status = new Status(res, (Element) elem.getElementsByTagName(
					"status").item(0), twitter);
			id = getChildText("id", elem);
			text = getChildText("text", elem);
			source = getChildText("source", elem);
			if ("".equals(source.trim()))
			{
				Element elementSource = (Element) elem.getElementsByTagName(
						"source").item(0);
				source = getTextContent("a", elementSource);

				sourceUrl = getAttribute("a", "href", elementSource);

			}
			createdAt = getChildDate("created_at", elem, twitter.myMicroBlog);
		}
		catch (Exception e)
		{
			System.out.println(e.getMessage());
		}
	}

	/**
	 * Return the created_at
	 * 
	 * @return created_at
	 * @since Twitter4J 1.1.0
	 */

	public Date getCreatedAt()
	{
		return this.createdAt;
	}

	/**
	 * Returns the id of the status
	 * 
	 * @return the id
	 */
	public String getId()
	{
		return this.id;
	}

	/**
	 * Returns the text of the status
	 * 
	 * @return the text
	 */
	public String getText()
	{
		return this.text;
	}

	/**
	 * Returns the source
	 * 
	 * @return the source
	 * @since Twitter4J 1.0.4
	 */
	public String getSource()
	{
		return this.source;
	}

	/**
	 * Test if the status is truncated
	 * 
	 * @return true if truncated
	 * @since Twitter4J 1.0.4
	 */
	public boolean isTruncated()
	{
		return isTruncated;
	}

	/**
	 * Returns the in_reply_tostatus_id
	 * 
	 * @return the in_reply_tostatus_id
	 * @since Twitter4J 1.0.4
	 */
	public long getInReplyToStatusId()
	{
		return inReplyToStatusId;
	}

	/**
	 * Returns the in_reply_user_id
	 * 
	 * @return the in_reply_tostatus_id
	 * @since Twitter4J 1.0.4
	 */
	public int getInReplyToUserId()
	{
		return inReplyToUserId;
	}

	/**
	 * Returns the in_reply_to_screen_name
	 * 
	 * @return the in_in_reply_to_screen_name
	 * @since Twitter4J 2.0.4
	 */
	public String getInReplyToScreenName()
	{
		return inReplyToScreenName;
	}

	/**
	 * returns The location's latitude that this tweet refers to.
	 * 
	 * @since Twitter4J 2.0.10
	 */
	public double getLatitude()
	{
		return latitude;
	}

	/**
	 * returns The location's longitude that this tweet refers to.
	 * 
	 * @since Twitter4J 2.0.10
	 */
	public double getLongitude()
	{
		return longitude;
	}

	/**
	 * Test if the status is favorited
	 * 
	 * @return true if favorited
	 * @since Twitter4J 1.0.4
	 */
	public boolean isFavorited()
	{
		return isFavorited;
	}

	private User user = null;

	/**
	 * Return the user
	 * 
	 * @return the user
	 */
	public User getUser()
	{
		return user;
	}

	/**
	 * 
	 * @since Twitter4J 2.0.10
	 */
	public boolean isRetweet()
	{
		return null != retweetDetails;
	}

	/**
	 * 
	 * @since Twitter4J 2.0.10
	 */
	public RetweetDetails getRetweetDetails()
	{
		return retweetDetails;
	}

	/* package */
	static List<Comment> constructStatuses(Response res, Weibo twitter)
			throws WeiboException
	{
		Document doc = res.asDocument();
		if (isRootNodeNilClasses(doc))
		{
			return new ArrayList<Comment>(0);
		}
		else
		{
			try
			{

				ensureRootNodeNameIs("comments", doc);

				NodeList list = doc.getDocumentElement().getElementsByTagName(
						"comment");

				int size = list.getLength();
				List<Comment> comments = new ArrayList<Comment>(size);
				for (int i = 0; i < size; i++)
				{
					Element status = (Element) list.item(i);
					comments.add(new Comment(res, status, twitter));
					
				}

				return comments;
			}
			catch (WeiboException te)
			{
				ensureRootNodeNameIs("nil-classes", doc); 
				return new ArrayList<Comment>(0);
			}
		}
	}

	@Override
	public int hashCode()
	{
		return id.hashCode();
	}

	@Override
	public boolean equals(Object obj)
	{
		if (null == obj)
		{
			return false;
		}
		if (this == obj)
		{
			return true;
		}
		return obj instanceof Comment && ((Comment) obj).id == this.id;
	}

	public String getSourceUrl()
	{
		return sourceUrl;
	}

	@Override
	public String toString()
	{
		return "Status{" + "createdAt=" + createdAt + ", id=" + id + ", text='"
				+ text + '\'' + ", source='" + source + '\'' + ", isTruncated="
				+ isTruncated + ", inReplyToStatusId=" + inReplyToStatusId
				+ ", inReplyToUserId=" + inReplyToUserId + ", isFavorited="
				+ isFavorited + ", inReplyToScreenName='" + inReplyToScreenName
				+ '\'' + ", latitude=" + latitude + ", longitude=" + longitude
				+ ", retweetDetails=" + retweetDetails + ", user=" + user + '}';
	}

	public Status getStatus()
	{
		return status;
	}

	public void setStatus(Status status)
	{
		this.status = status;
	}

}
