/**********
Copyright 1990 Regents of the University of California.  All rights reserved.
Author: 1986 Thomas L. Quarles
**********/

#ifndef IFSIMULATOR
#define IFSIMULATOR

  #ifdef _DEBUG
  #ifdef _MSC_VER
  #pragma message( "  parsing IFSim.H" )
  #endif /*_MSC_VER*/
  #endif /*_DEBUG*/

  #include "basic.h"
  #include "instance.h"

  /*
   * structure:   IFparm
   *
   *
   * The structure used to describe all values passed
   * between the front end and the simulator when there is any
   * possibility one argument of the function could have more
   * than one type.
   *
   * keyword is provided for the front end and is the token
   *    the user is expected to label the data with.
   *
   * id is an integer intended to uniquely identify the parameter
   *    to the simulator
   *
   * dataType is an integer which indicates the type of argument
   *    that must be passed for this parameter
   *
   * description is a longer description intended for help menus
   *    the description should all fit on one line, but should
   *    give a knowledgable user a good idea what the parameter is
   *    used for.
   */

  typedef struct sIFparm {
      char *keyword;
      int id;
      int dataType;
      char *description;
  } IFparm;

  /* Structure:   IFparseTree
   *
   * This structure is returned by the parser for a IF_PARSETREE valued
   * parameter and describes the information that the simulator needs
   * to know about the parse tree in order to use it.
   * It is expected that the front end will have a more extensive
   * structure which this structure will be a prefix of.
   *
   * Note that the function pointer is provided as a hook for
   * versions which may want to compile code for the parse trees
   * if they are used heavily.
   *
   */

  typedef struct sIFparseTree {
      int numVars;            /* number of variables used */
      int *varTypes;          /* array of types of variables */
      union uIFvalue * vars;  /* array of structures describing values */
      IITEXPORT(int, IFeval)(InstanceHandle,struct sIFparseTree*,double,double*,double*,double*);
  } IFparseTree;

  /*
   * Structure: IFanalysis
   *
   * This structure contains all the information available to the
   * front end about a particular analysis type.  The simulator will
   * present the front end with an array of pointers to these structures
   * which it will use to determine legal analysis types and parameters.
   *
   * Note to simulators:  As for IFdevice above, you pass an array of pointers
   * to these, so you can make this structure a prefix to a larger structure
   * which you use internally.
   *
   */

  typedef struct sIFanalysis {
      char *name;                 /* name of this analysis type */
      char *description;          /* description of this type of analysis */

      int numParms;               /* number of analysis parameter descriptors */
      struct sIFparm *analysisParms;      /* array  of analysis parameter descriptors */

  } IFanalysis;


  #ifdef XSPICE
    /* gtri - add - wbk - 10/11/90 - for structs referenced in IFdevice */
    #include "MIFtypes.h"
    #include "MIFdefs.h"
    #ifdef XSPICE_BUILD
      #include "xspice.h"
    #endif
    #include "MIFparse.h"
    #include "MIFcmdat.h"
    /* gtri - end - wbk - 10/11/90 */
  #endif

  #include "ftedata.h"

  /*
   *
   * types for IFnewUid
   *
   */

  #define UID_ANALYSIS 0x1
  #define UID_TASK 0x2
  #define UID_INSTANCE 0x4
  #define UID_MODEL 0x8
  #define UID_SIGNAL 0x10
  #define UID_OTHER 0x20


  /*
   * dataType values:
   *
   * Note:  These structures are put together by ORing together the
   *    appropriate bits from the fields below as is shown for the vector
   *    types.
   * IF_REQUIRED indicates that the parameter must be specified.
   *    The front end does not NEED to check for this, but can to save time,
   *    since failure to do so will cause the simulator to fail.
   * IF_SET indicates that the specified item is an input parameter.
   * IF_ASK indicates that the specified item is something the simulator
   *    can provide information about.
   * IF_SET and IF_ASK are NOT mutually exclusive.
   * if IF_SET and IF_ASK are both zero, it indicates a parameter that
   *    the simulator recoginizes are being a reasonable paremeter, but
   *    which this simulator does not implement.
   */

  #define IF_FLAG 0x1
  #define IF_INTEGER 0x2
  #define IF_REAL 0x4
  #define IF_COMPLEX 0x8
  #define IF_NODE 0x10
  #define IF_STRING 0x20
  #define IF_INSTANCE 0x40
  #define IF_PARSETREE 0x80

  /* indicates that for a query the integer field will have a selector
   * in it to pick a sub-field */
  #define IF_SELECT 0x800
  #define IF_VSELECT 0x400

  /* indicates a vector of the specified type */
  #define IF_DIAGRAM_GRAPH_NORETX  0x1000
  #define IF_DIAGRAM_CHART         0x2000
  #define IF_DIAGRAM_GRAPH         0x4000
  #define IF_DIAGRAM_ALL         ( IF_DIAGRAM_CHART | IF_DIAGRAM_GRAPH | IF_DIAGRAM_GRAPH_NORETX )
  #define RemoveDiagramType(x)   ( (x) & ~IF_DIAGRAM_ALL )
  #define IF_VECTOR 0x8000

  #define IF_FLAGVEC     (IF_FLAG|IF_VECTOR)
  #define IF_INTVEC      (IF_INTEGER|IF_VECTOR)
  #define IF_REALVEC     (IF_REAL|IF_VECTOR)
  #define IF_CPLXVEC     (IF_COMPLEX|IF_VECTOR)
  #define IF_NODEVEC     (IF_NODE|IF_VECTOR)
  #define IF_STRINGVEC   (IF_STRING|IF_VECTOR)
  #define IF_INSTVEC     (IF_INSTANCE|IF_VECTOR)

  #define IF_REQUIRED 0x4000

  #define IF_VARTYPES 0x80ff

  #define IF_SET 0x2000
  #define IF_ASK 0x1000

  /* If you AND with IF_UNIMP_MASK and get 0, it is recognized, but not
   * implemented
   */
  #define IF_UNIMP_MASK (~0xfff)

  /* Used by sensetivity to check if a parameter is or is not useful */
  #define IF_REDUNDANT	0x0010000
  #define IF_PRINCIPAL	0x0020000
  #define IF_AC		0x0040000
  #define IF_AC_ONLY	0x0080000
  #define IF_NOISE	0x0100000
  #define IF_NONSENSE	0x0200000

  #define IF_SETQUERY	0x0400000
  #define IF_ORQUERY	0x0800000
  #define IF_CHKQUERY	0x1000000

  /* For "show" command: do not print value in a table by default */
  #define IF_UNINTERESTING 0x2000000


  /*
   * Structure:    IFvalue
   *
   * structure used to pass the values corresponding to the above
   * dataType.  All types are passed in one of these structures, with
   * relatively simple rules concerning the handling of the structure.
   *
   * whoever makes the subroutine call allocates a single instance of the
   * structure.  The basic data structure belongs to you, and you
   * should arrange to free it when appropriate.
   *
   * The responsibilities of the data supplier are:
   * Any vectors referenced by the structure are to be malloc()'d
   * and are assumed to have been turned over to the recipient and
   * thus should not be re-used or free()'d.
   *
   * The responsibilities of the data recipient are:
   * scalar valued data is to be copied by the recipient
   * vector valued data is now the property of the recipient,
   * and must be free()'d when no longer needed.
   *
   * Character strings are a special case:  Since it is assumed
   * that all character strings are directly descended from input
   * tokens, it is assumed that they are static, thus nobody
   * frees them until the circuit is deleted, when the front end
   * may do so.
   *
   * EVERYBODY's responsibility is to be SURE that the right data
   * is filled in and read out of the structure as per the IFparm
   * structure describing the parameter being passed.  Programs
   * neglecting this rule are fated to die of data corruption
   *
   */

  /*
   * and of course, the standard complex data type
   */
  typedef struct sIFcomplex {
      double real;
      double imag;
  } IFcomplex;


  typedef union uIFvalue {
      int iValue;             /* integer or flag valued data */
      double rValue;          /* real valued data */
      IFcomplex cValue;       /* complex valued data */
      char *sValue;           /* string valued data */
      IFuid uValue;           /* UID valued data */
      IFnode nValue;          /* node valued data */
      IFparseTree *tValue;    /* parse tree */
      struct {
          int numValue;       /* length of vector */
          union {
              int *iVec;      /* pointer to integer vector */
              double *rVec;   /* pointer to real vector */
              IFcomplex *cVec;/* pointer to complex vector */
              char **sVec;    /* pointer to string vector */
              IFuid *uVec;    /* pointer to UID vector */
              IFnode *nVec;   /* pointer to node vector */
          }vec;
      }v;
  } IFvalue;

  #ifdef IIT
  extern char * EXPORT spice_malloc(InstanceHandle inst, int num);
  #endif

  /*
   * structure:  IFdevice
   *
   * This structure contains all the information available to the
   * front end about a particular device.  The simulator will
   * present the front end with an array of pointers to these structures
   * which it will use to determine legal device types and parameters.
   *
   * Note to simulators:  you are passing an array of pointers to
   * these structures, so you may in fact make this the first component
   * in a larger, more complex structure which includes other data
   * which you need, but which is not needed in the common
   * front end interface.
   *
   */

  /* gtri - modify - wbk - 10/11/90 - add entries to hold data required */
  /*                                  by new parser                     */

  typedef struct sIFdevice {
      char *name;                 /* name of this type of device */
      char *description;          /* description of this type of device */

      #if defined(IIT) || !defined(XSPICE)
          int *terms;                 /* number of terminals on this device */
          int *numNames;              /* number of names in termNames */
      #else
          int terms;                  /* number of terminals on this device */
          int numNames;               /* number of names in termNames */
      #endif
      char **termNames;           /* pointer to array of pointers to names */
				  /* array contains 'terms' pointers */


      #if defined(IIT) || !defined(XSPICE)
          int *numInstanceParms;      /* number of instance parameter descriptors */
      #else
          int numInstanceParms;       /* number of instance parameter descriptors */
      #endif
      IFparm *instanceParms;      /* array  of instance parameter descriptors */

      #if defined(IIT) || !defined(XSPICE)
          int *numModelParms;         /* number of model parameter descriptors */
      #else
          int numModelParms;          /* number of model parameter descriptors */
      #endif
      IFparm *modelParms;         /* array  of model parameter descriptors */

      #if defined(IIT) || !defined(XSPICE)
          int flags;			/* DEV_ */
      #endif

      #ifdef XSPICE
          void (*cm_func)(Mif_Private_t *);  /* pointer to code model function */

          int num_conn;               /* number of code model connections */
          Mif_Conn_Info_t  *conn;     /* array of connection info for mif parser */

          int num_param;              /* number of parameters = numModelParms */
          Mif_Param_Info_t *param;    /* array of parameter info for mif parser */

          int num_inst_var;              /* number of instance vars = numInstanceParms */
          Mif_Inst_Var_Info_t *inst_var; /* array of instance var info for mif parser */
      #endif

  } IFdevice;
  /* gtri - end - wbk - 10/11/90 */

  #ifdef XSPICE
     #define IFdevice_Berkeley_default_for_XSPICE_params \
             NULL, 0, NULL, 0, NULL, 0, NULL
  #else
     #define IFdevice_Berkeley_default_for_XSPICE_params 
  #endif

  /*
   * Structure: IFsimulator
   *
   * This is what we have been leading up to all along.
   * This structure describes a simulator to the front end, and is
   * returned from the SIMinit command to the front end.
   * This is where all those neat structures we described in the first
   * few hundred lines of this file come from.
   *
   */
  typedef enum FileMode    {EXT_NONE, EXT_OUT, EXT_ERR} FileMode;
  typedef enum ExecuteMode {EXE_NONE, EXE_CLI, EXE_DOT_OPTION} ExecuteMode;
  typedef enum ExecuteResult {EXER_IGNORED, EXE_FAIL, EXER_SUCCEED} ExecuteResult;

  int EXPORT GetNodeAndCompList(CKTcircuit*             ckt,
                                ANALYSIS_TYPE           analysis_type,
                                AnalysisNode**          node_array,
                                int*                    num_nodes,
                                AnalysisComponents**    component_array,
                                int*                    num_components
                               );

  struct circ* EXPORT GetCircuitList(InstanceHandle handle);
  struct circ* EXPORT GetCurrentCircuit(InstanceHandle handle);

  typedef struct sIFsimulator {
      char *simulator;                /* the simulator's name */
      char *description;              /* description of this simulator */
      char *version;                  /* version or revision level of simulator*/

      int numDevices;                 /* number of device types supported */
      IFdevice **devices;             /* array of device type descriptors */

      int numAnalyses;                /* number of analysis types supported */
      IFanalysis **analyses;          /* array of analysis type descriptors */

      int numNodeParms;               /* number of node parameters supported */
      IFparm *nodeParms;              /* array of node parameter descriptors */

      int numSpecSigs;        /* number of special signals legal in parse trees */
      char **specSigs;        /* names of special signals legal in parse trees */

      /* create new circuit */
      IITEXPORT(int, newCircuit)(InstanceHandle handle,
                                 GENERIC ** ckt
                                );

      /* destroy old circuit's data structures*/
      IITEXPORT(int, deleteCircuit)(GENERIC * ckt);

      /* create new node */
      IITEXPORT(int, newNode)(GENERIC*  ckt,
                              GENERIC** node,
                              IFuid name
                             );

      /* create ground node */
      IITEXPORT(int, groundNode)(GENERIC*  ckt,
                                 GENERIC** node,
                                 IFuid     name
                                );

      /* bind a node to a terminal */
      IITEXPORT(int, bindNode)(GENERIC* ckt,
                               GENERIC* instance,
                               int      terminal,
                               GENERIC* node
                              );

      /* find a node by name */
      IITEXPORT(int, findNode)(GENERIC*  ckt,
                               GENERIC** node,
                               IFuid     name
                              );

      /* find the node attached to a terminal */
      IITEXPORT(int, instToNode)(GENERIC*  ckt,
                                 GENERIC*  instance,
                                 int       terminal,
                                 GENERIC** node,
                                 IFuid*    nodeName
                                );

      /* set a parameter on a node */
      IITEXPORT(int, setNodeParm)(GENERIC* ckt,
                                  GENERIC* node,
                                  int      param_number,
                                  IFvalue* value,
                                  IFvalue* selector
                                 );

      /* ask a question about a node */
      IITEXPORT(int, askNodeQuest)(GENERIC* ckt,
                                   GENERIC* node,
                                   int      param_number,
                                   IFvalue* value,
                                   IFvalue* selector
                                  );

      /* delete a node from the circuit */
      IITEXPORT(int, deleteNode)(GENERIC* ckt,
                                 GENERIC* node
                                );

      /* create new instance */
      IITEXPORT(int, newInstance)(GENERIC*   ckt,
                                  GENERIC*   model,
                                  GENERIC**  instance,
                                  IFuid      name
                                 );

      /* set a parameter on an instance */
      IITEXPORT(int, setInstanceParm)(GENERIC* ckt,
                                      GENERIC* instance,
                                      int      param_number,
                                      IFvalue* value,
                                      IFvalue* selector
                                     );

      /* ask a question about an instance */
      IITEXPORT(int, askInstanceQuest)(GENERIC*  ckt,
                                       GENERIC*  instance,
                                       int       which,
                                       IFvalue*  value,
                                       IFvalue*  selector
                                      );

      /* find a specific instance */
      IITEXPORT(int, findInstance)(GENERIC*   ckt,
                                   int*       type,
                                   GENERIC**  instance,
                                   IFuid      name,
                                   GENERIC*   model,
                                   IFuid      model_name
                                  );

      /* delete an instance from the circuit */
      IITEXPORT(int, deleteInstance)(GENERIC* ckt,
                                     GENERIC* instance
                                    );

      /* create new model */
      IITEXPORT(int, newModel)(GENERIC*  ckt,
                               int       type,
                               GENERIC** model,
                               IFuid     name
                              );

      /* set a parameter on a model */
      IITEXPORT(int, setModelParm)(GENERIC*  ckt,
                                   GENERIC*  model,
                                   int       param_number,
                                   IFvalue*  value,
                                   IFvalue*  selector
                                  );

      /* ask a questions about a model */
      IITEXPORT(int, askModelQuest)(GENERIC*  ckt,
                                    GENERIC*  model,
                                    int       which,
                                    IFvalue*  value,
                                    IFvalue*  selector
                                   );

      /* find a specific model */
      IITEXPORT(int, findModel)(GENERIC*  ckt,
                                int*      type,
                                GENERIC** model,
                                IFuid     name
                               );

      /* delete a model from the circuit*/
      IITEXPORT(int, deleteModel)(GENERIC* ckt,
                                  GENERIC* model
                                 );

      /* create a new task */
      IITEXPORT(int, newTask)(GENERIC*  ckt,
                              GENERIC** task,
                              IFuid     name
                             );

      /* create new analysis within a task */
      IITEXPORT(int, newAnalysis)(GENERIC*  ,
                                  int       ,
                                  IFuid,
                                  GENERIC** ,
                                  GENERIC*
                                 );

      /* set a parameter on an analysis  */
      IITEXPORT(int, setAnalysisParm)(GENERIC* ,
                                      GENERIC* ,
                                      int      ,
                                      IFvalue* ,
                                      IFvalue*
                                     );

      /* ask a question about an analysis */
      IITEXPORT(int, askAnalysisQuest)(GENERIC* ,
                                       GENERIC* ,
                                       int      ,
                                       IFvalue* ,
                                       IFvalue*
                                      );

      /* find a specific analysis */
      IITEXPORT(int, findAnalysis)(GENERIC*   ,
                                   int*       ,
                                   GENERIC**  ,
                                   IFuid      ,
                                   GENERIC*   ,
                                   IFuid
                                  );

      /* find a specific task */
      IITEXPORT(int, findTask)(GENERIC*   ,
                               GENERIC**  ,
                               IFuid
                              );

      /* delete a task */
      IITEXPORT(int, deleteTask)(GENERIC* ,
                                 GENERIC*
                                );


      IITEXPORT(int, doAnalyses)(GENERIC*  ,
                                 int       ,
                                 GENERIC*
                                );

  #if defined(IIT) || !defined(XSPICE)
      IITEXPORT(char *, nonconvErr)(GENERIC*,char *); /* return nonconvergence error */
  #endif

  #ifdef EWB_IIT
	  IITEXPORT(int, getMatrixElement)(GENERIC*, GENERIC*, GENERIC*, double**);
	  IITEXPORT(int, getVoltage)(GENERIC*, GENERIC*, GENERIC*, double*);
	  IITEXPORT(int, addVoltage)(GENERIC*, GENERIC*, double);
	  IITEXPORT(int, getCurrent)(GENERIC*, GENERIC*, double*);
	  IITEXPORT(int, addCurrent)(GENERIC*, GENERIC*, GENERIC*, double);
	  IITEXPORT(int, setBreakPoint)(GENERIC*, double);
	  IITEXPORT(int, setRelativeBreakPoint)(GENERIC*, double);
	  IITEXPORT(int, changeCapacitorValue)(GENERIC*, GENERIC*, double);
	  IITEXPORT(int, changeInductorValue)(GENERIC*, GENERIC*, double);
	  IITEXPORT(int, isFirstIteration)(GENERIC*, int*);
	  IITEXPORT(int, setMaxStep)(GENERIC*, double);
  #endif

#ifdef IIT
      IITEXPORT(int, executeShellCmd)(InstanceHandle handle,
                                      FileMode       mode,
                                      char*          command
                                     );

      IITEXPORT(int, GetNodeAndCompList)(
                               CKTcircuit*             ckt,
                               ANALYSIS_TYPE           analysis_type,
                               AnalysisNode**          node_array,
                               int*                    num_nodes,
                               AnalysisComponents**    component_array,
                               int*                    num_components
                             );

      IITEXPORT(struct circ*, GetCircuitList)(InstanceHandle handle);
      IITEXPORT(struct circ*, GetCurrentCircuit)(InstanceHandle handle);
#endif
  } IFsimulator;


typedef struct PlotterInfo
{
        double *xlims, *ylims;          /* The size of the screen. */
        char *xname, *plotname;         /* What to label things. */
    	char *hcopy;                    /* The raster file. */
        int nplots;                     /* How many plots there will be. */
        double xdelta, ydelta;          /* Line increments for the scale. */
        int gridtype;                   /* The grid type */
        int plottype;                   /*  and the plot type. */
        char *xlabel, *ylabel;          /* Labels for axes. */
        int xtype, ytype;               /* The types of the data graphed. */
        char *pname;
        char *commandline;              /* For xi_zoomdata() */
} PlotterInfo;


  /*
   * Structure: IFfrontEnd
   *
   * This structure provides the simulator with all the information
   * it needs about the front end.  This is the entire set of
   * front end and back end related routines the simulator
   * should know about.
   *
   */

  typedef struct sIFfrontEnd {
      /* create a new UID in the circuit */
      IITEXPORT(int, IFnewUid)(GENERIC*,IFuid*,IFuid,char*,int,GENERIC**);

      /* create a new UID in the circuit */
      IITEXPORT(int, IFdelUid)(GENERIC*,IFuid,int);

      /* should we stop now? */
      IITEXPORT(int, IFpauseTest)(InstanceHandle inst);

      /* what time is it? */
      IITEXPORT(double, IFseconds)(InstanceHandle inst);

      /* output an error or warning message */
      IITEXPORT(int, IFerror)(InstanceHandle inst, int,char*,IFuid*);

      /* start pointwise output plot */
      IITEXPORT(int, OUTpBeginPlot)(GENERIC*,GENERIC*,IFuid,IFuid,int,
                                    int,IFuid*,int,GENERIC**);
                              
      /* data for pointwise plot */
      IITEXPORT(int, OUTpData)(InstanceHandle inst,GENERIC*,IFvalue*,IFvalue*);
                              
      /* start windowed output plot */
      IITEXPORT(int, OUTwBeginPlot)(InstanceHandle inst, GENERIC*,GENERIC*,IFuid,IFuid,int,
                                   int,IFuid*,int,GENERIC**);
                              
      /* independent vector for windowed plot */
      IITEXPORT(int, OUTwReference)(InstanceHandle inst, GENERIC*,IFvalue*,GENERIC**);
                              
      /* data for windowed plot */
      IITEXPORT(int, OUTwData)(InstanceHandle inst, GENERIC*,int,IFvalue*,GENERIC*);
                              
      /* signal end of windows */
      IITEXPORT(int, OUTwEnd)(InstanceHandle inst, GENERIC*);

      /* end of plot */
      IITEXPORT(int, OUTendPlot)(InstanceHandle inst, GENERIC*);
                              
      /* start nested domain */
      IITEXPORT(int, OUTbeginDomain)(InstanceHandle inst, GENERIC*,char *,int,IFvalue*);
                              
      /* end nested domain */
      IITEXPORT(int, OUTendDomain)(InstanceHandle inst, GENERIC*);
                              
      IITEXPORT(int, OUTattributes)(InstanceHandle inst, GENERIC *,char*,int,IFvalue*);

      #ifdef EWB_IIT
      /* setup devices */
	  IITEXPORT(int, EWBsetup)(void);

      /* load devices */
	  IITEXPORT(int, EWBload)(double timeStep, double time);

      /* load devices (ac) */
	  IITEXPORT(int, EWBacload)(double freq);

      /* accept timepoint */
	  IITEXPORT(int, EWBaccept)(double time, double *bp);

      /* convergence test */
	  IITEXPORT(int, EWBconvTest)(int *noncon);

      /* convert the name of a SPICE  to a EWB node */
      IITEXPORT(char*, EWBconvNode)(char* nodename);
      #endif

     #ifdef IIT
      /* Access EWB's message table */
      IITEXPORT(char*, EWBMsg)(InstanceHandle handle, unsigned long);
     #endif

     #ifdef IIT
      IITEXPORT(int, shellCmdOutput)(InstanceHandle handle, FileMode mode, char* command);
     #endif

      IITEXPORT(ExecuteResult, cmdExecute)(InstanceHandle handle, ExecuteMode mode, char* command);

      /* Initialize the grap*/
      IITEXPORT(int, graphInit)(InstanceHandle handle, void** context, PlotterInfo* pi);

      /* Graph a datavector */
      IITEXPORT(int, graphPoints)(InstanceHandle handle, void* context, DataVector* v, DataVector* xs, int nostart);

      /* Uninitialize the graph */
      IITEXPORT(int, graphCleanup)(InstanceHandle handle, void** context);

      /* Register a new UID in the circuit */
      IITEXPORT(int, RegisterUid)(InstanceHandle handle, IFuid* uid, int type);

      /* Unregister a new UID in the circuit */
      IITEXPORT(int, UnRegisterUid)(InstanceHandle handle, IFuid* uid);

      /* Allocate memory */
      IITEXPORT(void*, Malloc)(InstanceHandle handle, int size);

      /* Reallocate memory */
      IITEXPORT(void*, ReAlloc)(InstanceHandle handle, void* mem, int newsize);

      /* Free memory */
      IITEXPORT(void, Free)(InstanceHandle handle, void* uid);

  } IFfrontEnd;

  char* XSpiceMsg(InstanceHandle handle, unsigned long msgid);

  /* flags for the first argument to IFerror */
  #define ERR_WARN  0x1
  #define ERR_FATAL 0x2
  #define ERR_PANIC 0x4
  #define ERR_INFO  0x8

      /* valid values for the second argument to doAnalyses */

      /* continue the analysis from where we left off */
  #define RESUME 0
      /* start everything over from the beginning of this task*/
  #define RESTART 1
      /* abandon the current analysis and go on the the next in the task*/
  #define SKIPTONEXT 2

  #define OUT_SCALE_LIN   1
  #define OUT_SCALE_LOG   2

#endif /*IFSIMULATOR*/
